package com.agilex.healthcare.directscheduling.mdws.translator;


import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import com.agilex.healthcare.directscheduling.domain.AppointmentSlotFilter;
import com.agilex.healthcare.directscheduling.domain.AppointmentTimeSlot;
import com.agilex.healthcare.directscheduling.domain.AppointmentTimeSlots;
import com.agilex.healthcare.directscheduling.domain.BookAppointment;
import com.agilex.healthcare.directscheduling.domain.BookAppointmentEmail;
import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.AppointmentTO;
import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.HospitalLocationTO;
import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.TaggedHospitalLocationArray;
import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.TaggedText;
import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.TimeSlotArray;
import com.agilex.healthcare.directscheduling.mdws.generatedwsdl.schedulingservice.TimeSlotTO;
import com.agilex.healthcare.directscheduling.utils.DateHelper;
import com.agilex.healthcare.directscheduling.validator.MdwsResponseException;

public class AppointmentTimeSlotTranslator {

    public AppointmentTimeSlots translateFromHospitalLocationTO(TaggedHospitalLocationArray hospitalLocationArrayTO, AppointmentSlotFilter appointmentSlotFilter) {
        AppointmentTimeSlots retValue = new AppointmentTimeSlots();


        if (hospitalLocationArrayTO.getCount() <= 0) {
            throw new MdwsResponseException(MdwsResponseException.UserMessages.INVALID_DATA_RETURNED, "hospitalLocationArrayTO was null");
        }
        if (hospitalLocationArrayTO.getFault() != null)
            throw new MdwsResponseException(hospitalLocationArrayTO.getFault());

        List<HospitalLocationTO> hospitalLocationListTo = hospitalLocationArrayTO.getLocations().getHospitalLocationTO();
        if (hospitalLocationListTo == null || hospitalLocationListTo.isEmpty()) {
            throw new MdwsResponseException(MdwsResponseException.UserMessages.INVALID_DATA_RETURNED, "hospitalLocationListTo is empty or null");
        }

        HospitalLocationTO hospitalLocationTO = hospitalLocationListTo.get(0);
        String clinicId = appointmentSlotFilter.getClinicId();
        String name = hospitalLocationTO.getName();
        String appointmentLength = hospitalLocationTO.getAppointmentLength();
        String clinicDisplayStartTime = hospitalLocationTO.getClinicDisplayStartTime();
        String displayIncrements = hospitalLocationTO.getDisplayIncrements();
        boolean askForCheckIn = hospitalLocationTO.isAskForCheckIn();
        String maxOverbooksPerDay = hospitalLocationTO.getMaxOverbooksPerDay();

        setAppointmentTimeSlots(retValue, hospitalLocationTO, appointmentSlotFilter);

        retValue.setClinicId(clinicId);
        retValue.setClinicName(name);
        retValue.setAppointmentLength(appointmentLength);
        retValue.setClinicDisplayStartTime(clinicDisplayStartTime);
        retValue.setDisplayIncrements(displayIncrements);
        retValue.setAskForCheckIn(askForCheckIn);
        retValue.setMaxOverbooksPerDay(maxOverbooksPerDay);

        TaggedText stopCodeTag = hospitalLocationTO.getStopCode();
        TaggedText creditStopCodeTag = hospitalLocationTO.getCreditStopCode();
        TaggedText specialtyTag = hospitalLocationTO.getSpecialty();

        if (stopCodeTag != null) {
            if (stopCodeTag.getFault() != null) throw new MdwsResponseException(stopCodeTag.getFault());
            retValue.setStopCode(stopCodeTag.getTag());
        }
        if (creditStopCodeTag != null) {
            if (creditStopCodeTag.getFault() != null) throw new MdwsResponseException(creditStopCodeTag.getFault());
            retValue.setCreditCode(creditStopCodeTag.getTag());
        }
        if (specialtyTag != null) {
            if (specialtyTag.getFault() != null) throw new MdwsResponseException(specialtyTag.getFault());
            retValue.setSpecialty(specialtyTag.getTag());
        }

        if (retValue.size() > 0) {
            calculateRemainingOverbookingsPerDay(retValue);
        }

        return retValue;
    }

    private void setAppointmentTimeSlots(AppointmentTimeSlots appointmentTimeSlots, HospitalLocationTO hospitalLocationTO, AppointmentSlotFilter appointmentSlotFilter) {
        TimeSlotArray availability = hospitalLocationTO.getAvailability();
        if (availability == null)
            return;
        //Perform Error Checking
        if (availability.getFault() != null)
            throw new MdwsResponseException(availability.getFault());
        if (availability.getSlots() == null || availability.getSlots().getTimeSlotTO() == null)
            return;

        Iterator<TimeSlotTO> itTimeSlotTO = availability.getSlots().getTimeSlotTO().iterator();
        while (itTimeSlotTO.hasNext()) {
            TimeSlotTO timeSlotTO = itTimeSlotTO.next();
            //Perform Error Checking
            if (timeSlotTO.getFault() != null)
                throw new MdwsResponseException(timeSlotTO.getFault());

            Date startDate = DateHelper.toDate(timeSlotTO.getStart());
            Date endDate = DateHelper.toDate(timeSlotTO.getEnd());
            Date filterStartDate = appointmentSlotFilter.getStartDate();
            Date filterEndDate = appointmentSlotFilter.getEndDate();

            if (isTimeSlotWithinSpecifiedDateRange(startDate, endDate, filterStartDate, filterEndDate)) {
//				if(timeSlotTO.isAvailable()){
                String text = timeSlotTO.getText();
                boolean available = timeSlotTO.isAvailable();

                AppointmentTimeSlot appointmentTimeSlot = new AppointmentTimeSlot();
                appointmentTimeSlot.setStartDateTime(startDate);
                appointmentTimeSlot.setEndDateTime(endDate);
                appointmentTimeSlot.setBookingStatus(text);
                appointmentTimeSlot.setAvailability(available);
                appointmentTimeSlots.add(appointmentTimeSlot);
//				}
            }
        }
    }

    private boolean isTimeSlotWithinSpecifiedDateRange(Date startDate, Date endDate, Date desiredStartDate, Date desiredEndDate) {
        return (startDate.equals(desiredStartDate) || startDate.after(desiredStartDate)) && (endDate.equals(desiredEndDate) || endDate.before(desiredEndDate));
    }

    protected void calculateRemainingOverbookingsPerDay(AppointmentTimeSlots appointmentTimeSlots) {
        int remainingOverBookings;
        HashMap<String, Integer> dayMap = new HashMap<String, Integer>();
        SimpleDateFormat df = new SimpleDateFormat("MM/dd/yyyy");

        //
        // What we are doing is taking the maximum allowed over bookings per day for a clinic
        // and calculating the remaining allowed over bookings by looping through all the time slots,
        // adding up the current number of over bookings and subtracting that number from the maximum
        // allowed over bookings to determine the remaining allowed over bookings.
        //

        //
        // Calculate the number of remaining allowed over bookings for each time slot by day.
        //
        if (!(appointmentTimeSlots.getMaxOverbooksPerDay().equalsIgnoreCase(""))) {
            int maximumAllowedOverBookings = Integer.parseInt(appointmentTimeSlots.getMaxOverbooksPerDay());
            Iterator<AppointmentTimeSlot> timeSlotIterator = appointmentTimeSlots.iterator();
            while (timeSlotIterator.hasNext()) {
                AppointmentTimeSlot timeSlot = timeSlotIterator.next();

                String date = df.format(timeSlot.getStartDateTime());

                if (dayMap.containsKey(date) == false) {
                    remainingOverBookings = new Integer(maximumAllowedOverBookings);
                    dayMap.put(date, remainingOverBookings);
                } else {
                    remainingOverBookings = dayMap.get(date).intValue();
                }

                remainingOverBookings -= calculateOverbookings(timeSlot);
                if (remainingOverBookings < 0)
                    remainingOverBookings = 0;

                dayMap.put(date, new Integer(remainingOverBookings));
            }

            //
            // Set the remaining allowed over bookings for the day based on what
            // was calculated above.
            //
            timeSlotIterator = appointmentTimeSlots.iterator();
            while (timeSlotIterator.hasNext()) {
                AppointmentTimeSlot timeSlot = timeSlotIterator.next();

                String date = df.format(timeSlot.getStartDateTime());

                String remainingAllowed = dayMap.get(date).toString();
                timeSlot.setRemainingAllowedOverBookings(remainingAllowed);
            }

        }
    }

    private int calculateOverbookings(AppointmentTimeSlot timeSlot) {
        String textFieldValue;
        char c;

        textFieldValue = timeSlot.getBookingStatus();

        if (textFieldValue.length() == 1) {
            c = textFieldValue.charAt(0);
            //
            // "X" means "Don't book anything in this time slot, including overbookings.
            //
            if (c >= 'A' && c < 'X') {
                //
                // An "A" means that there was one overbooking.  "A" is 65 in ascii, so if we
                // subtract 64 from the character value, we will get the number of overbookings
                // booked for this slot.
                //
                return c - 64;
            }
        }

        return 0;
    }

//----------------------------------------------------------------------------------------------------------------------

    public BookAppointment translate(AppointmentTO appointmentTO, String siteCode) {
        BookAppointmentEmail retValue = new BookAppointmentEmail();
        if (appointmentTO == null)
            throw new MdwsResponseException(MdwsResponseException.UserMessages.INVALID_DATA_RETURNED, "AppointmentTO was null");
        if (appointmentTO.getFault() != null)
            throw new MdwsResponseException(appointmentTO.getFault());

        HospitalLocationTO clinic = appointmentTO.getClinic();
        String timestamp = appointmentTO.getTimestamp();
        Date dtDateTime = DateHelper.parseVistaDateTime(timestamp);
        String purpose = appointmentTO.getPurpose();

        if (clinic != null) {
            //Perform Error Checking
            if (clinic.getFault() != null)
                throw new MdwsResponseException(clinic.getFault());
            String id = clinic.getId();
            retValue.setClinicId(id);
            retValue.setFacilityId(siteCode);
        }

        retValue.setDateTime(dtDateTime);
        retValue.setPurpose(purpose);
        return retValue;
    }
}
